package hu.afghangoat.widgets;

import hu.afghangoat.ConfigParser;
import hu.afghangoat.exceptions.InvalidDistanceException;
import hu.afghangoat.exceptions.InvalidGoalPositionException;
import hu.afghangoat.helpers.FlatMapCoordinate;
import hu.afghangoat.helpers.GPSCoordinate;
import hu.afghangoat.helpers.ImageHelper;
import hu.afghangoat.helpers.MathHelpers;
import hu.afghangoat.simulators.MouseEventSimulator;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;

import static hu.afghangoat.helpers.GPSCoordinate.pixelsPerMeter;

/**
 * @class GuessMap
 * @brief A Swing component that displays an interactive map for guessing locations.
 *
 * The GuessMap class displays a map where the user can guess a coordinate, zoom and pan in the map.
 * The map also computes the guessed accuracy.
 *
 * @implements MouseWheelListener
 * @implements MouseListener
 * @implements MouseMotionListener
 */
public class GuessMap extends JLabel implements MouseWheelListener, MouseListener, MouseMotionListener {

    /**
     * @brief The minimum allowed zoom factor for the map.
     *
     * This constant defines how far the map can be zoomed out.
     * A smaller value means the map can appear more zoomed out.
     */
    public static final double minZoomFactor=0.5;

    /**
     * @brief The maximum allowed zoom factor for the map.
     *
     * This constant defines how far the map can be zoomed in.
     * A larger value means the map can appear more zoomed in.
     */
    public static final double maxZoomFactor=1.0;

    /**
     * @brief A stored reference of the map image as a Swing ImageIcon.
     */
    private ImageIcon icon;

    /**
     * @brief A stored reference of the map image as a Swing Image
     *
     * Will be used at the drawing phase.
     */
    private Image image;

    /**
     * @brief A stored reference of the unscaled map image as a Swing ImageIcon.
     *
     */
    private ImageIcon originalIcon;

    /**
     * @brief Specifies, how much the original map image should be rescaled.
     *
     * Will be stored and be used for the map rendering.
     */
    private double scale=1.0;

    /**
     * @brief The zooming factor which will be accounted in the drawing function.
     *
     * This will scale the map accordingly.
     */
    private double zoomFactor = 1;

    /**
     * @brief The value of the previous zoom factor stored.
     *
     * Needed for proper graphics drawing.
     */
    private double prevZoomFactor = 1;

    /**
     * @brief Indicates whether the user zooms.
     */
    private boolean zoomer;

    /**
     * @brief Indicates whether the user drags.
     */
    private boolean dragger;

    /**
     * @brief Indicates whether the user released the mouse left button.
     */
    private boolean released;

    /**
     * @brief The X offset of the map from the map panel frame top-left position.
     */
    private double xOffset = 0;

    /**
     * @brief The Y offset of the map from the map panel frame top-left position.
     */
    private double yOffset = 0;

    /**
     * @brief The X difference based on the previous drag state when drawing of the map from the map panel frame top-left position.
     */
    private int xDiff;

    /**
     * @brief The Y difference based on the previous drag state when drawing of the map from the map panel frame top-left position.
     */
    private int yDiff;

    /**
     * @brief Stores the mouse left clock positions.
     */
    private Point startPoint;

    /**
     * @brief Stores the computed width of the map based on the supplied scale.
     */
    private double selfImgWidth;

    /**
     * @brief Stores the computed height of the map based on the supplied scale.
     */
    private double selfImgHeight;

    /**
     * @brief A getter which returns whether the user has already placed a marker.
     */
    public boolean isPlacedMarker() {
        return placedMarker;
    }

    /**
     * @brief Stores whether the user already placed a marker or not.
     */
    boolean placedMarker=false;

    /**
     * @brief Stores a reference of the marker image as a Swing ImagerIcon variable.
     */
    private ImageIcon markerIcon;

    /**
     * @brief Stores a reference of the marker image as a Swing Image variable.
     */
    private Image markerImage;

    /**
     * @brief Stores the computed width of the marker image based on the supplied scale.
     */
    private double markerImgWidth;

    /**
     * @brief Stores the computed height of the marker image based on the supplied scale.
     */
    private double markerImgHeight;

    /**
     * @brief Stores where the user has set the marker on the X axis.
     *
     * Is -1 if not set.
     */
    private double markerPosX=-1;

    /**
     * @brief Stores where the user has set the marker on the Y axis.
     *
     * Is -1 if not set.
     */
    private double markerPosY=-1;

    /**
     * @brief Stores the distance based on the marker position.
     *
     * Is -1 if not set.
     */
    private double distanceAccuracy=-1.0; //-1.0 if not guessed, >=0.0 if guessed.

    /**
     * @brief Stores whether the user has guessed or not.
     *
     * Is false if not set.
     */
    private boolean guessed=false;

    /**
     * @brief Returns the distance accuracy of the player's guess.
     *
     * This method computes and returns the accuracy of the user's guess
     * based on the distance between the guessed location and the actual target.
     * If the accuracy has not yet been computed, an error will be thrown.
     *
     * @return The distance accuracy in pixels, converted from meters.
     *
     * @note This method marks the guess as complete and triggers a repaint.
     * @warning If `distanceAccuracy` is -1.0, it indicates an uninitialized state.
     */
    public double getDistanceAccuracy() throws InvalidDistanceException {

        if(distanceAccuracy==-1.0){
            throw new InvalidDistanceException("For some reason, no distanceAccuracy set.");
        }
        guessed=true;
        repaint();
        //System.out.println("distance accuracy: "+distanceAccuracy);

        return distanceAccuracy* GPSCoordinate.pixelsPerMeter;
    }

    /**
     * @brief Returns the raw guessed distance.
     *
     * @return The raw distance in meters.
     */
    public double getGuessedDistance(){
        double dx=markerPosX-goalPosX;
        double dy=markerPosY-goalPosY;

        //System.out.println("Diff: "+dx+" | "+dy);

        return Math.sqrt(dx*dx+dy*dy);

    }

    /**
     * @brief Stores the generated goal position's X segment.
     *
     * Is -1.0 if not set.
     */
    private double goalPosX=-1.0;

    /**
     * @brief Stores the generated goal position's Y segment.
     *
     * Is -1.0 if not set.
     */
    private double goalPosY=-1.0;

    /**
     * @brief Sets the goal position to an (x,y) float pair.
     *
     * @param gPX The X segment of the goal position.
     * @param gPY The Y segment of the goal position.
     */
    public void setGoalPos(double gPX, double gPY){
        guessed=false;

        distanceAccuracy=-1.0;
        placedMarker=false;

        markerPosX=-1;
        markerPosY=-1;

        goalPosX=gPX;
        goalPosY=gPY;

        //System.out.println("GoalPos: "+goalPosX+" | "+goalPosY);
    }

    /**
     * @brief Sets the goal position to a FlatMapCoordinate.
     *
     * @param coords The input FlatMapCoordinate
     */
    public void setGoalPos(FlatMapCoordinate coords){
        setGoalPos(coords.getPosX(),coords.getPosY());
    }

    /**
     * @brief Sets the marker position to an (x,y) float pair.
     *
     * @param nX The X segment of the marker position.
     * @param nY The Y segment of the marker position.
     */
    public void setMarkerTo(double nX, double nY) throws InvalidGoalPositionException{
        if(goalPosX==-1.0||goalPosY==-1.0){
            throw new InvalidGoalPositionException("No goal was set or its position is invalid");
        } else {
            placedMarker=true;


            System.out.println("Marker placed");

            markerPosX=(int)(nX);
            markerPosY=(int)(nY);

            distanceAccuracy= getGuessedDistance();


        }


    }

    /**
     * @brief The constructor of the guessMap which takes in the scale which will affect the map render
     *
     * @param _scale The scale of the map.
     */
    public GuessMap(double _scale){
        scale=_scale;

        originalIcon = new ImageIcon(ConfigParser.getConfigPath() + "uni_map.png");

        selfImgWidth=(double) originalIcon.getIconWidth();
        selfImgHeight=(double)originalIcon.getIconHeight();

        //System.out.println(selfImgWidth+"|"+selfImgHeight);

        image = originalIcon.getImage();


        icon = new ImageIcon(image);
       // setIcon(icon); //THIS double draws, but removing it will set the label to show nothing.


        markerIcon = new ImageIcon(ConfigParser.getConfigPath() + "assets/location.png");
        markerImgWidth=(double) markerIcon.getIconWidth();
        markerImgHeight=(double)markerIcon.getIconHeight();

        markerImage=markerIcon.getImage();//.getScaledInstance((int)(markerImgWidth*scale),(int)(markerImgHeight*scale), Image.SCALE_SMOOTH);

        //setSize(new Dimension((int)(selfImgWidth*scale),(int)(selfImgHeight*scale)));
        setOpaque(true);

        this.addListeners();
    }

    /**
     * @brief Sets up and initializes the Swing event listeners.
     */
    private void addListeners(){
        addMouseWheelListener(this);
        addMouseMotionListener(this);
        addMouseListener(this);
    }

    /**
     * @brief Returns the size of the image frame after the scaling applied.
     *
     * @return The size of the image frame in Dimension unit.
     */
    @Override
    public Dimension getPreferredSize() {
        return new Dimension((int)(selfImgWidth*scale), (int)(selfImgHeight*scale));
    }

    /**
     * @brief Makes the per-frame painting
     *
     * Also makes the resizing, repositioning transformations.
     *
     * If the user has already guessed, it draws the marker and if the program is in the check phase it shows a tangent line from the marker to the goal location.
     *
     * @param g The inherited Graphics Swing tag.
     */
    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);

        Graphics2D g2 = (Graphics2D) g;

        AffineTransform at = getAffineTransform();

        g2.setTransform(at);
        g2.drawImage(image, 0, 0, this);

        // draw marker if placed
        if (markerPosX != -1 && markerPosY != -1) {
            g2.drawImage(markerImage, (int)(markerPosX - markerImgWidth / 2.0), (int)(markerPosY - markerImgHeight), this);
        }

        if(markerPosX!=-1&&markerPosY!=-1){
            g2.drawImage(markerImage, (int)(markerPosX-markerImgWidth/2.0), (int)(markerPosY-markerImgHeight), this);

            if(guessed==true){
                int x = 50, y = 50;
                int z = 200, w = 200;

                g2.setColor(Color.DARK_GRAY);
                g2.setStroke(new BasicStroke(5));
                g2.drawLine((int)(markerPosX), (int)(markerPosY), (int)goalPosX, (int)goalPosY);

            }
        }
    }

    /**
     * @brief Applies the correct offset and zoom based on the mouse input.
     *
     * @return The AffineTransform with the applied offset and zoom.
     */
    private AffineTransform getAffineTransform() {
        double imageWidth = selfImgWidth * zoomFactor;
        double imageHeight = selfImgHeight * zoomFactor;

        double minXOffset = Math.min(0, getWidth() - imageWidth);
        double minYOffset = Math.min(0, getHeight() - imageHeight);

        double clampedX = Math.max(minXOffset, Math.min(0, xOffset));
        double clampedY = Math.max(minYOffset, Math.min(0, yOffset));

        AffineTransform at = new AffineTransform();
        at.translate(clampedX, clampedY);
        at.scale(zoomFactor, zoomFactor);
        return at;
    }

    /**
     * @brief Sets the zoom based on the mouse wheel movement.
     *
     * @param e The mouse wheel movement event
     */
    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {

        zoomer = true;

        //Zoom in
        if (e.getWheelRotation() < 0) {
            zoomFactor *= 1.1;

            if(zoomFactor>maxZoomFactor){
                zoomFactor=maxZoomFactor;
            }
            repaint();
        }
        //Zoom out
        if (e.getWheelRotation() > 0) {
            zoomFactor /= 1.1;

            if(zoomFactor<minZoomFactor){
                zoomFactor=minZoomFactor;
            }
            xOffset/= 1.1;
            yOffset/= 1.1;

            repaint();
        }

        repaint();

        MouseEventSimulator view_fixer = new MouseEventSimulator();
        view_fixer.simulateDrag(50,50,49,49,this);
    }

    /**
     * @brief A helper function which needs to be called when zooming in.
     */
    public void zoomIn(){
        repaintWithParent();
    }

    /**
     * @brief A helper function which needs to be called when zooming in or out.
     *
     * The zooming will not be visible if this does not get called.
     */
    public void repaintWithParent(){
        Container parent = this.getParent();
        if (parent != null) {
            parent.repaint();
        }
        this.repaint();
    }

    /**
     * @brief A helper function which needs to be called when zooming out.
     */
    public void zoomOut(){
        repaintWithParent();
    }

    /**
     * @brief A helper function which helps calculating the zoom and pan factor based on the offset.
     *
     * @param mousePos The position of the mouse in one axis.
     *
     * @param offset The offset position on one axis.
     *
     * @return The modified position.
     */
    public double zoomAndPanFactor(double mousePos, double offset){
        return  (mousePos - offset) / zoomFactor;
    }

    /**
     * @brief Handles the mouse click event.
     *
     * Places the guess marker at the position of the mouse event click.
     *
     * @param e The click mouse event.
     */
    @Override
    public void mouseClicked(MouseEvent e) {

        if(guessed==true){
            return;
        }

        int mouseX = e.getX();
        int mouseY = e.getY();

        double imageX = (mouseX - xOffset) / zoomFactor;
        double imageY = (mouseY - yOffset) / zoomFactor;

// Clamp so clicks outside the image don’t cause issues
        imageX = MathHelpers.clampd(imageX, 0, image.getWidth(null) - 1);
        imageY = MathHelpers.clampd(imageY, 0, image.getHeight(null) - 1);
        //imageY = Math.max(0, Math.min(image.getHeight(null)-1, imageY));

        //System.out.println("Mouse over image pixel: " + imageX + ", " + imageY);

        try {
            setMarkerTo(imageX,imageY);
        } catch (InvalidGoalPositionException ex) {
            System.out.println("Failed to set marker, no valid goal pos defined!");
        }
        //System.out.println("Mouse clicked at: X:"+mouseX+" Y:"+mouseY+" !");

        //Consume click so no random giga-dragging occurs.
        e.consume();
        repaint();

        //MouseEventSimulator view_fixer = new MouseEventSimulator();
        //view_fixer.simulateDrag(50,50,49,49,this);
    }

    /**
     * @brief Handles the mouse press event.
     *
     * Starts tracking the mouse position using the startPoint field.
     *
     * @param e The press mouse event.
     */
    @Override
    public void mousePressed(MouseEvent e) {
        released = false;
        startPoint = e.getPoint(); // relative to the component
    }

    /**
     * @brief Sets the offset based on the mouse drag  movement.
     *
     * @param e The mouse drag movement event
     */
    @Override
    public void mouseDragged(MouseEvent e) {
        if (startPoint == null) return;

        int dx = e.getX() - startPoint.x;
        int dy = e.getY() - startPoint.y;

        xOffset += dx;
        yOffset += dy;

        double imageWidth = selfImgWidth * zoomFactor;
        double imageHeight = selfImgHeight * zoomFactor;

        double minXOffset = Math.min(0, getWidth() - imageWidth);
        double minYOffset = Math.min(0, getHeight() - imageHeight);

        xOffset = Math.max(minXOffset, Math.min(0, xOffset));
        yOffset = Math.max(minYOffset, Math.min(0, yOffset));

        startPoint = e.getPoint(); // update for next drag
        repaint();
    }



    /**
     * @brief Handles the mouse release event.
     * <p>
     * Signals, that the user released the mouse.
     * Also repaints the map for safety.
     *
     * @param e The release mouse event.
     */
    @Override
    public void mouseReleased(MouseEvent e) {
        released = true;
        startPoint = null; // reset
        dragger = false;
        repaint();
    }

    /**
     * @brief Internal, unused Swing methods which needs to be implemented.
     * <p>
     * For now, it is an empty implementation.
     *
     * @param e A mouse event
     */
    @Override
    public void mouseEntered(MouseEvent e) {
        //hovered = true;
        //repaint();
    }

    /**
     * @brief Internal, unused Swing methods which needs to be implemented.
     * <p>
     * For now, it is an empty implementation.
     *
     * @param e A mouse event
     */
    @Override
    public void mouseExited(MouseEvent e) {
       // hovered = false;
       // repaint();
    }

    /**
     * @brief Internal, unused Swing methods which needs to be implemented.
     * <p>
     * For now, it is an empty implementation.
     *
     * @param e A mouse event
     */
    @Override
    public void mouseMoved(MouseEvent e) {

    }
}
