package hu.afghangoat.widgets;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.RoundRectangle2D;

/**
 * @class FancyButton
 * @brief A button which has a gradient background and theme-wise.
 */
public class FancyButton extends JButton {

    /**
     * @brief The base background color of the button.
     */
    private Color baseColor = new Color(255, 255, 255);

    /**
     * @brief The hover background color of the button.
     */
    private Color hoverColor = new Color(204, 0, 0);

    /**
     * @brief Tracks whether the button is hovered.
     */
    private boolean hover = false;

    /**
     * @brief Sets a text for the button in the constructor and sets the styles needed for gradiency.
     *
     * @param text The text display of the button.
     */
    public FancyButton(String text) {

        super(text);


        setContentAreaFilled(false);
        setFocusPainted(false);
        setForeground(Color.BLACK);
        setFont(new Font("Segoe UI", Font.BOLD, 16));
        setBorder(new EmptyBorder(10, 20, 10, 20));
        setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));

        // Hover effect
        addMouseListener(new MouseAdapter() {
            @Override
            public void mouseEntered(MouseEvent e) {
                hover = true;
                repaint();
            }

            @Override
            public void mouseExited(MouseEvent e) {
                hover = false;
                repaint();
            }
        });
    }

    /**
     * @brief Sets an image icon in the constructor.
     *
     * @param tempLangIcon The icon which needs to be set.
     */
    public FancyButton(ImageIcon tempLangIcon) {
        super(tempLangIcon);
        setContentAreaFilled(false);
        setFocusPainted(false);
        setForeground(Color.BLACK);
        setFont(new Font("Segoe UI", Font.BOLD, 16));
        setBorder(new EmptyBorder(3, 3, 3, 3));
        setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));

        // Hover effect
        addMouseListener(new MouseAdapter() {
            @Override
            public void mouseEntered(MouseEvent e) {
                hover = true;
                repaint();
            }

            @Override
            public void mouseExited(MouseEvent e) {
                hover = false;
                repaint();
            }
        });
    }

    /**
     * @brief Overrides the paint component and renders the button with the custom specs.
     *
     * @param g The graphics object which is needed for the rendering.
     */
    @Override
    protected void paintComponent(Graphics g) {
        Graphics2D g2 = (Graphics2D) g.create();
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        Color fill = hover ? hoverColor : baseColor;
        g2.setColor(fill);

        Shape rectBorder = new RoundRectangle2D.Float(0, 0, getWidth(), getHeight(), 0, 0);
        Color startColor = hover ? hoverColor.brighter() : baseColor.brighter();
        Color endColor = hover ? hoverColor.darker() : baseColor.darker();

        GradientPaint gp = new GradientPaint(0, 0, startColor, 0, getHeight(), endColor);
        g2.setPaint(gp);
        g2.fill(rectBorder);

        super.paintComponent(g);
        g2.dispose();
    }
}