package hu.afghangoat.views;

import hu.afghangoat.ConfigParser;
import hu.afghangoat.LangParser;
import hu.afghangoat.widgets.BackButton;
import hu.afghangoat.widgets.GradientPanel;
import hu.afghangoat.widgets.StyleWidget;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

/**
 * @class SettingsPanel
 * @brief The panel provides UI interface for the visual and game settings.
 *
 */
public class SettingsPanel extends GradientPanel implements ViewInterface, ActionListener {

    /**
     * @brief Holds a general settings title for the page.
     */
    private JLabel label = new JLabel("SETINGS_DESCRIPTION");

    /**
     * @brief This label indicates that after it, the user may change the theme of the layout.
     */
    private JLabel bg_label = new JLabel("SETINGS_DESCRIPTION_BG");

    /**
     * @brief Clicking the "back" button will take the user back to the main menu.
     */
    private BackButton backBtn;

    /**
     * @brief Holds all the style options. Will be used in constructing the UI.
     */
    private ArrayList<StyleWidget> styleOptions = new ArrayList<>();

    /**
     * @brief This method handles what should happen when the language is changed.
     */
    @Override
    public void langChanged(){
        label.setText(LangParser.getLang("SETINGS_DESCRIPTION"));
        bg_label.setText(LangParser.getLang("SETINGS_DESCRIPTION_BG"));
        backBtn.setText(LangParser.getLang("BACK"));
    }

    /**
     * @brief Finds and returns a widget by the theme name.
     *
     * @param name The name if the widget.
     *
     * @return The widget by same name or null if not found.
     */
    private StyleWidget getWidgetByName(String name){
        for(int i=0;i<styleOptions.size();i++){
            if(styleOptions.get(i).getName().equals(name)){
                return styleOptions.get(i);
            }
        }
        return null;
    }

    /**
     * @brief Builds a button selector group for the widgets.
     *
     * @param mainWindow The parent window.
     */
    private void makeButtonGroup(MainWindow mainWindow){
        ButtonGroup group = new ButtonGroup();
        JPanel buttonPanel = new JPanel();

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = GridBagConstraints.RELATIVE;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.NONE;
        //label_gbc.weightx = 1.0;
        gbc.insets = new Insets(0, 10, 25, 10); // spacing

        ActionListener listener = e -> {
            JRadioButton source = (JRadioButton) e.getSource();
            //System.out.println("Selected: " + source.getText());

            StyleWidget curWidget = getWidgetByName(source.getText());
            if(curWidget==null){
                throw new RuntimeException("NULL style was set. This is unexpected.");
            } else {
                GradientPanel.setGradientColors(curWidget.getBgGradientColor1(),curWidget.getBgGradientColor2());
                mainWindow.setColorForAll(curWidget.getFontColor());
                repaint();
            }

        };

        for(int i=0;i<styleOptions.size();i++){
            JRadioButton option = new JRadioButton(styleOptions.get(i).getName());

            if(i==0){
                option.setSelected(true);
            }

            group.add(option);
            option.addActionListener(listener);
            buttonPanel.add(option);
        }

        add(buttonPanel,gbc);

    }

    /**
     * @brief This constructor sets up the layout of the panel and sets the event listeners. Also takes the parent window account.
     *
     * @param mainWindow The parent window of the panel.
     */
    public SettingsPanel(MainWindow mainWindow) {
        super(new GridBagLayout());

        styleOptions = mainWindow.cp.getThemes(ConfigParser.getConfigPath()+"styles.xml");

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = GridBagConstraints.RELATIVE;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.NONE;
        //label_gbc.weightx = 1.0;
        gbc.insets = new Insets(0, 10, 25, 10); // spacing

        backBtn = new BackButton(mainWindow,"BACK");

        label.setFont(new Font("Segoe UI", Font.BOLD, 30));
        add(label,gbc);

        add(bg_label,gbc);

        makeButtonGroup(mainWindow);


        add(backBtn,gbc);
    }

    /**
     * @brief Empty action listener, needs to be implemented.
     *
     * @param e The event of the action.
     * @deprecated Moved logic to the constructor.
     */
    @Override
    @Deprecated
    public void actionPerformed(ActionEvent e) {

    }
}