package hu.afghangoat.views;

import hu.afghangoat.ConfigParser;
import hu.afghangoat.LangParser;
import hu.afghangoat.blockchain.BlockChainHelper;
import hu.afghangoat.exceptions.InvalidGoalPositionException;
import hu.afghangoat.widgets.GradientPanel;

import javax.swing.*;
import java.awt.*;

/**
 * @class MainWindow
 * @brief Handles the view switching logic between the games.
 *
 * Also keeps track of the current language and overrides the panels with a new language if needed.
 *
 */
public class MainWindow extends JFrame {

    /**
     * @brief This layout will store all of the panels which are needed during the UI interactions.
     */
    private CardLayout cardLayout;

    /**
     * @brief The reference to the main menu panel.
     */
    private JPanel mainPanel;

    /**
     * @brief The default size of the window.
     */
    public static final Dimension DIVINE_SIZE =new Dimension(640, 480);

    /**
     * @brief A reference to a config parser which helps to reach the config files.
     */
    public ConfigParser cp;

    /**
     * @brief A reference to an inner language parser which can help in loading new language files.
     */
    private LangParser rf;

    /**
     * @brief A setter which helps to set a new language parser when needed.
     *
     * @param lp The new language parsers.
     */
    public void setRF(LangParser lp){

        rf=lp;
    }

    /**
     * @brief The reference to the end game panel.
     */
    private EndGamePanel ePanel;

    /**
     * @brief The reference to the leaderboard display panel.
     */
    private TopListPanel topPanel;

    /**
     * @brief The reference to the game logic panel.
     */
    private GamePanel gPanel;

    /**
     * @brief The reference to the settings panel.
     */
    private SettingsPanel settingsPanel;

    /**
     * @brief The inner first hash of the genesis block.
     *
     * This will be set for the initial previous hash for the top list.
     */
    public static int TOP_LIST_HASH=37181;

    /**
     * @brief Handles the top list save-load logic as well as the HTML format conversion.
     */
    private BlockChainHelper topListHelper = new BlockChainHelper(TOP_LIST_HASH);

    /**
     * @brief A wrapper method which will delegate the request to the blockchain helper field.
     *
     * @param name The entry name.
     * @param score The entry score.
     */
    public void addTopListEntry(String name, int score){
        topListHelper.addBlock(name, score);
        topListHelper.saveToFile(ConfigParser.getConfigPath()+ConfigParser.TOP_LIST_FILE_SAVE);
    }

    /**
     * @brief An event which verifies the blockchain on need.
     *
     * (After the top list is loaded.)
     */
    public void onTopListMessageReceived(){


        if(topListHelper.verifyHash()==false){
            topPanel.setTopListEntriesLabel(LangParser.getLang("INTEGRITY_FAILED"));
            System.out.println("INTEGRITY_FAILED");
        } else {

            String entries = topListHelper.toHTMLSorted(TopListPanel.VISIBLE_TOP_ENTRIES);
            topPanel.setTopListEntriesLabel(entries);
        }

    }

    /**
     * @brief Delegates a request game to the game panel.
     */
    public void requestStartRound() throws InvalidGoalPositionException {
        gPanel.startRound();
    }

    /**
     * @brief The default constructor handles the initialization of the panels and loading the default language.
     *
     * It also commences the harvesting of the data from the basic config file.
     */
    public MainWindow() {
        cp = new ConfigParser();

        rf = new LangParser(ConfigParser.getLangsPath()+cp.getCurrentLang()+".lang");

        topListHelper.loadFromFile(ConfigParser.getConfigPath()+ConfigParser.TOP_LIST_FILE_SAVE);

        setTitle(ConfigParser.getTitle());
        setMinimumSize(DIVINE_SIZE);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        setLocationRelativeTo(null);

        cardLayout = new CardLayout();
        mainPanel = new JPanel(cardLayout);

        mainPanel.add(new MenuPanel(this), "MENU");
        mainPanel.add(new HowToPanel(this), "HOWTO");
        mainPanel.add(new CreditsPanel(this), "CREDITS");


        settingsPanel = new SettingsPanel(this);
        mainPanel.add(settingsPanel, "SETTINGS");

        gPanel = new GamePanel(this);
        mainPanel.add(gPanel, "GAME");

        ePanel=new EndGamePanel(this);
        mainPanel.add(ePanel, "ENDGAME");

        topPanel=new TopListPanel(this);
        mainPanel.add(topPanel,"TOP_PANEL");

        changeLangForAll();

        add(mainPanel);
        pack();
    }

    /**
     * @brief Delegates the score to the end game panel.
     *
     * @param score The score which need to be passed to the end game panel.
     */
    public void showScore(int score){
        if(ePanel!=null){
            ePanel.showScore(score);
        }

    }

    /**
     * @brief Requests a Swing repaint and frees all the panels from the existing realm.
     */
    public void clearFrame(){
        getContentPane().removeAll();
        revalidate();
        repaint();
    }

    /**
     * @brief Shows a panel based in a requested panel name.
     *
     * @param name The name of the requested panel.
     */
    public void showView(String name) {
        cardLayout.show(mainPanel, name);
    }

    /**
     * @brief Requests a language changing for all the panels.
     */
    public void changeLangForAll(){
        for (Component comp : mainPanel.getComponents()) {
            if (comp instanceof ViewInterface) {
                ((ViewInterface) comp).langChanged();
            }
        }
    }


    /**
     * @brief Requests a theme repaint for all the panels.
     *
     * @param newColor The new font color for the panels.
     */
    public void setColorForAll(Color newColor){
        for (Component comp : mainPanel.getComponents()) {
            if (comp instanceof GradientPanel) {
                ((GradientPanel) comp).setTextColor(newColor);
            }
        }
    }
}

