package hu.afghangoat.views;

import hu.afghangoat.ConfigParser;
import hu.afghangoat.LangParser;
import hu.afghangoat.exceptions.InvalidDistanceException;
import hu.afghangoat.exceptions.InvalidGoalPositionException;
import hu.afghangoat.helpers.*;
import hu.afghangoat.simulators.MouseEventSimulator;
import hu.afghangoat.widgets.*;
import org.jdom2.Element;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import static hu.afghangoat.ConfigParser.USES_GPS;
import static hu.afghangoat.helpers.FlatMapParser.XML_IMAGE_TAG;

/**
 * @class GamePanel
 * @brief The panel which commences the main game logic.
 *
 * The player can scroll a panorama image and based on that, guess on the map besides it.
 *
 */
public class GamePanel extends GradientPanel implements ViewInterface, ActionListener {

    //private JLabel label = new JLabel("CREDITS_DESCRIPTION");

    /**
     * @brief Clicking the "back" button will take the user back to the main menu.
     */
    private BackButton backBtn;

    /**
     * @brief Stores a draggable map element which the user can use to place a marker to a guessable location.
     *
     */
    private GuessMap map;

    /**
     * @brief This label will display the objectives language-wise to the user.
     */
    JLabel guessLabel;

    /**
     * @brief Displays how much points the player has.
     */
    JLabel pointsLabel;

    /**
     * @brief Displays which round the player is in.
     */
    JLabel roundLabel;

    /**
     * @brief Displays either the "Guess" or the "Next round" option to the player language-wise.
     */
    private JButton nextGameBtn;

    /**
     * @brief A helper field which will be used in the checking of the coordinate correctness the player has guessed.
     */
    private FlatMapParser flatMapParser = new FlatMapParser();

    /**
     * @brief The name of the current to-be-displayed image.
     */
    private String imgName= "1.jpg";

    /**
     * @brief This panel displays the location image and the guess map in a border layout.
     */
    private JPanel imageAndMapPanel;

    /**
     * @brief Tracks how much points the player has accumulated.
     */
    private int totalPoints=0;

    /**
     * @brief Tracks which round is the player in. -1 if the game is not started or ended.
     */
    private int currentRound=-1;

    /**
     * @brief This sets that how long should a game take.
     */
    public static final int MAX_ROUNDS=5;

    /**
     * @brief This stores image and location pairs for each round.
     *
     * In each round, a new image will be displayed.
     */
    private List<Element> currentEntries=new ArrayList<>();//getShuffledLocations

    /**
     * @brief The frame which allows the image to be scrolled like a panorama image.
     */
    private ScrollableImageFrame imageLabel=null;

    /**
     * @brief Tracks how much the player had missed in meters in each round.
     */
    private double mDist=0.0;

    /**
     * @brief Sets the falloff range where after a larger miss, the user gets no points.
     */
    public static final double MAX_DISTANCE_WHERE_POINTS_COUNT=100.0;

    /**
     * @brief Tracks the root window.
     */
    private MainWindow mainEndWindow;

    /**
     * @brief Sets the game parameters to the default values, also starts the first round.
     *
     * Uses the flatMapParser to get some random images for guessing.
     */
    public void startRound() throws InvalidGoalPositionException {

        currentRound=-1;
        totalPoints=0;
        currentEntries=flatMapParser.getShuffledLocations(MAX_ROUNDS);

        nextRound(0.0);
    }

    /**
     * @brief Can be used to calculate how many points should the player be awarded based on his accuracy.
     *
     * @return the amount of points from 0-MAX_DISTANCE_WHERE_POINTS_COUNT in a linear falloff range.
     */
    private int calculateFalloffPoints(double missedDistance){
        if(missedDistance>MAX_DISTANCE_WHERE_POINTS_COUNT){
            return 0;
        } else {
            return (int)(MAX_DISTANCE_WHERE_POINTS_COUNT-missedDistance);
        }

    }

    /**
     * @brief Starts the next round.
     *
     * If no round was started, it sets the guess button visible and sets the proper point label texts to their default value.
     *
     * @param missedDistance The missed distance in the previous round.
     */
    private void nextRound(double missedDistance) throws InvalidGoalPositionException {

        nextGameBtn.setVisible(false);
        System.out.println("Preparing next round...");

        if(currentRound!=-1){
            //Add points
            totalPoints+=calculateFalloffPoints(missedDistance);

        }
        if(pointsLabel!=null){
            System.out.println(LangParser.getLang("POINTS_LABEL")+": "+totalPoints);
            pointsLabel.setText(LangParser.getLang("POINTS_LABEL")+": "+totalPoints);
        }

        if(imageLabel==null){
            System.out.println("Imagelabel is null. This should not occur.");
            return;
        }

        //Check for end game
        currentRound++;
        if(roundLabel!=null){
            roundLabel.setText(LangParser.getLang("ROUNDS_LABEL")+": "+currentRound+" / "+MAX_ROUNDS);
        }

        if(currentRound>=Math.min(MAX_ROUNDS,currentEntries.size())){
            endGame();
            return;
        }

        //System.out.println(currentEntries.size()+" GG");
        imgName = (String)currentEntries.get(currentRound).getChild(XML_IMAGE_TAG).getText();

        GPSReader reader = new GPSReader();


        imageAndMapPanel.remove(imageLabel);
        imageLabel = new ScrollableImageFrame(new ScrollableImage(ConfigParser.getImagePath() + imgName, 400, 600));
        imageAndMapPanel.add(imageLabel);
        imageAndMapPanel.revalidate();
        imageAndMapPanel.repaint();

        if(ConfigParser.USES_GPS==true){
            try {
                System.out.println("Reading EXIF info...");
                GPSCoordinate gps = reader.extractGps(new File(ConfigParser.getImagePath() + imgName));
                if (gps != null) {
                    System.out.println("Latitude: " + gps.getGPSLatitude());
                    System.out.println("Longitude: " + gps.getGPSLongitude());

                    FlatMapCoordinate coords = gps.toFlatMapCoordinate();
                    map.setGoalPos(coords);
                    map.setMarkerTo(10,10);
                } else {
                    System.out.println("No GPS data found.");
                }
            } catch (Exception e){
                System.out.println("Failed to get EXIF info!");
            }
        } else {
            FlatMapCoordinate coords = flatMapParser.getOfflineCoordsForImg(imgName);
            if(coords==null){
                System.out.println("Failed to get coordinates for supplied image: "+imgName);
            } else {
                map.setGoalPos(coords);

            }

        }

        map.setMarkerTo(10,10);
    }

    /**
     * @brief Reroutes the player to the end game screen where the final score is shown.
     */
    private void endGame(){
        if(mainEndWindow!=null){
            mainEndWindow.showScore(totalPoints);
            mainEndWindow.showView("ENDGAME");
        }

    }

    /**
     * @brief This method handles what should happen when the language is changed.
     */
    @Override
    public void langChanged(){
        //label.setText(LangParser.getLang("CREDITS_DESCRIPTION"));
        backBtn.setText(LangParser.getLang("BACK"));
        guessLabel.setText(LangParser.getLang("GUESS_METERS"));
        nextGameBtn.setText(LangParser.getLang("NEXT_GAME"));
        pointsLabel.setText(LangParser.getLang("POINTS_LABEL"));
        roundLabel.setText(LangParser.getLang("ROUNDS_LABEL"));

    }

    private void guessed(double missedDistance){
        if(map.isPlacedMarker()==true){ //Csak ha már lerakta a markert, van értelme mutatni a guess-t.
            guessLabel.setText(missedDistance+"m");
            mDist=missedDistance;

            nextGameBtn.setVisible(true);

            map.repaint();
            MouseEventSimulator view_fixer = new MouseEventSimulator();
            view_fixer.simulateDrag(50,50,49,49,map);

        }

    }

    /**
     * @brief This constructor sets up the layout of the panel and sets the event listeners. Also takes the parent window account.
     *
     * @param mainWindow The parent window of the panel.
     */
    public GamePanel(MainWindow mainWindow){

        super(new GridBagLayout());

        mainEndWindow=mainWindow;

        imageAndMapPanel = new JPanel(new GridBagLayout());
        imageAndMapPanel.setOpaque(false);

        //Image frame
        imageLabel = new ScrollableImageFrame();
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 2;
        gbc.gridy = 0;

        gbc.insets = new Insets(10, 10, 10, 10);
        gbc.weightx = 1.0;
        gbc.weighty = 2.0;
        gbc.fill = GridBagConstraints.BOTH;
        imageAndMapPanel.add(imageLabel, gbc);

        GridBagConstraints guessUIGBC = new GridBagConstraints();
        guessUIGBC.gridx = 0;
        guessUIGBC.gridy = 0;
        guessUIGBC.gridwidth = 1;
        guessUIGBC.fill = GridBagConstraints.HORIZONTAL;
        guessUIGBC.weightx = 1.0;
        guessUIGBC.anchor = GridBagConstraints.CENTER;
        guessUIGBC.insets = new Insets(2, 2, 2, 2);

        JPanel guessUI = new JPanel(new GridBagLayout());

        guessLabel = new JLabel("GUESS_METERS");
        pointsLabel= new JLabel("POINTS_LABEL");
        roundLabel=new JLabel("ROUNDS_LABEL");

        map = new GuessMap(0.1);
        map.setGoalPos(10.0,10.0);

        guessUIGBC.gridx = 0;
        guessUIGBC.weightx = 1.0;
        guessUIGBC.fill = GridBagConstraints.HORIZONTAL;

        guessUIGBC.gridy = 0;
        guessUI.add(roundLabel, guessUIGBC);

        guessUIGBC.gridy = 1;
        guessUI.add(pointsLabel, guessUIGBC);

        guessUIGBC.gridy = 2;
        guessUI.add(guessLabel, guessUIGBC);

        // Map
        guessUIGBC.gridy = 3;
        guessUIGBC.fill = GridBagConstraints.BOTH;
        guessUIGBC.weighty = 1.0;
        guessUI.add(map, guessUIGBC);

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth=3;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        gbc.fill = GridBagConstraints.BOTH;
        imageAndMapPanel.add(guessUI,gbc);


        add(imageAndMapPanel, gbc);

        // === Middle: Buttons in a row ===
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;

        JButton prevBtn = new FancyButton("<-");

        gbc.gridx = 0;
        gbc.gridy = 1;
        add(prevBtn, gbc);

        JButton nextBtn = new FancyButton("->");
        gbc.gridx = 1;
        add(nextBtn, gbc);

        JButton guessBtn = new FancyButton("Guess");
        gbc.gridx = 2;
        add(guessBtn, gbc);

        // === Bottom: Back button ===
        backBtn = new BackButton(mainWindow, "BACK");
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.gridwidth = 2; // center across the full width
        gbc.anchor = GridBagConstraints.CENTER;
        add(backBtn, gbc);

        nextGameBtn = new FancyButton("NEXT_GAME");
        gbc.gridx = 2;
        gbc.gridwidth = 1; // center across the full width
        add(nextGameBtn, gbc);

        //Event listeners
        prevBtn.addActionListener(e -> {
            imageLabel.scrollRight();
            //System.out.println("Right");
        });

        nextBtn.addActionListener(e -> {
            imageLabel.scrollLeft();
            //System.out.println("Left");
        });

        guessBtn.addActionListener(e -> {
            try {
                this.guessed(map.getDistanceAccuracy());
            } catch (InvalidDistanceException ex) {
                System.out.println("Invalid distance computed! This should not have occurred!");
            }
        });

        nextGameBtn.addActionListener(e -> {

            try {
                nextRound(mDist);
            } catch (InvalidGoalPositionException ex) {
                System.out.println("Goal position is set to invalid!");
            }
        });
    }

    /**
     * @brief Empty action listener, needs to be implemented.
     *
     * @param e The event of the action.
     * @deprecated Moved logic to the constructor.
     */
    @Override
    @Deprecated
    public void actionPerformed(ActionEvent e) {

    }
}