package hu.afghangoat.helpers;


import com.drew.lang.annotations.Nullable;
import hu.afghangoat.ConfigParser;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaderXSDFactory;

import javax.xml.XMLConstants;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

/**
 * @class FlatMapParser
 * @brief A utility class which allows to parse FlatMap coordinates from designated XML config files.
 *
 * Can also be instantiated for getting multiple variants of configured classes.
 *
 */
public class FlatMapParser {

    /**
     * @brief The file name in the config directory which stores the offline flatmap coordinates for the game.
     */
    public static final String OFFLINE_COORDS_FILE = "offline_coords.xml";

    /**
     * @brief The file name in the config directory which stores the offline flatmap coordinate schema for the game.
     */
    public static final String OFFLINE_COORDS_FILE_VALIDATION = "offline_coords_schema.xsd";

    /**
     * @brief An XML tag constants which will be used in a program. This marks the (photographed) image path corresponding to a set of coordinates.
     */
    public static final String XML_IMAGE_TAG="ImageName";

    /**
     * @brief An XML tag constants which will be used in a program. This marks the location corresponding to a set of coordinates.
     */
    public static final String XML_LOCATION_TAG="Location";

    /**
     * @brief An XML tag constants which will be used in a program. This marks the root element of the entries.
     */
    public static final String XML_ROOT_TAG="CampusGuesser";

    /**
     * @brief An XML tag constants which will be used in a program. This marks the X position corresponding to a set of coordinates.
     */
    public static final String XML_X_POSITION_TAG="X";

    /**
     * @brief An XML tag constants which will be used in a program. This marks the Y position corresponding to a set of coordinates.
     */
    public static final String XML_Y_POSITION_TAG="Y";

    /**
     * @brief Stores the read-in locations with coordinates, images and X,Y cartesian coordinate components.
     */
    private List<Element> locations = new ArrayList<>();

    /**
     * @brief Returns an array of N shuffled coordinates from the read-in bank of entries.
     *
     * @param maxEntries How much entries should be returned.
     *
     * @return The shuffled N locations in an array.
     */
    public List<Element> getShuffledLocations(int maxEntries) {
        int entriesLength=locations.size();
        List<Element> shuffled = new ArrayList<>(locations);
        Collections.shuffle(shuffled, new Random());

        if(maxEntries>=entriesLength){
            return shuffled;
        } else {
            List<Element> limitedShuffled = new ArrayList<>();
            for(int i=0;i<maxEntries;i++){
                limitedShuffled.add(shuffled.get(i));
            }

            return limitedShuffled;
        }

    }

    /**
     * @brief A constructor which loads the entries from the hardcoded file.
     */
    public FlatMapParser(){
        try {
            // Point to your schema
            File schemaFile = new File(ConfigParser.getConfigPath() + OFFLINE_COORDS_FILE_VALIDATION);
            XMLReaderXSDFactory factory = new XMLReaderXSDFactory(schemaFile);
            SAXBuilder saxBuilder = new SAXBuilder(factory);

            Document document = saxBuilder.build(new File(ConfigParser.getConfigPath() + OFFLINE_COORDS_FILE));
            Element root = document.getRootElement();

            if(validateRootElement(root)==false){
                System.out.println("Invalid root element found in "+OFFLINE_COORDS_FILE+". Ignoring.");
            } else {
                locations = root.getChildren(XML_LOCATION_TAG);
            }



        } catch (Exception e) {
            System.out.println("Failed to parse the offline coordinate config holder XML file for the images. Consider this a failure. (Malformed XML or missing file) ");
            System.out.println("Path: "+ConfigParser.getConfigPath() + OFFLINE_COORDS_FILE_VALIDATION);
        }
    }

    /**
     * @brief Checks whether the name of the root element is correct.
     *
     * Since no schema is defined, extra checks are needed.
     *
     * @param root The root element
     *
     * @return Whether the root element integrity is correct.
     */
    private boolean validateRootElement(Element root){
        if(true==XML_ROOT_TAG.equals((String)root.getName())){
            return true;
        }
        return false;
    }

    /**
     * @brief Gives the flatmap coordinates for a give image.
     *
     * @param img_name The name of the image where the coordinates are needed.
     *
     * @return The flatmap position of the image.
     */
    public FlatMapCoordinate getOfflineCoordsForImg(String img_name) {
        try {
            for (Element loc : locations) {
                if(img_name.equals((String)loc.getChild(XML_IMAGE_TAG).getText()) == true){
                    double x = Double.parseDouble(loc.getChildText(XML_X_POSITION_TAG));
                    double y = Double.parseDouble(loc.getChildText(XML_Y_POSITION_TAG));
                    return new FlatMapCoordinate(x,y);
                }

            }


        } catch (Exception e) {
            System.out.println("Got invalid or null value in the offline coordinate registry.");
        }
        return null;
    }
}
