package hu.afghangoat.blockchain;

import java.util.Arrays;
import java.util.Objects;

/**
 * @class Block
 * @brief A single block contains some amount of transactions.
 *
 * Needed for the Top List logic
 *
 */
public class Block {

    /**
     * @brief The hash of the previous block.
     */
    private int prevHash;

    /**
     * @brief The list of transactions.
     */
    private Transaction[] transactions;

    /**
     * @brief The basic constructor of a block which takes in a previous hash and some transactions.
     *
     * @param prevHash The hash of the previous block.
     *
     * @param transactions The list of the transactions.
     */
    public Block(int prevHash, Transaction[] transactions) {
        this.prevHash = prevHash;
        this.transactions = transactions;
    }

    /**
     * @brief Returns the transactions of the block.
     *
     * @return the transactions of the block.
     */
    public Transaction[] getTransactions() {
        return transactions;
    }

    /**
     * @brief Sets the transactions of the block.
     *
     * @param transactions The new transactions which need to be set.
     */
    public void setTransactions(Transaction[] transactions) {
        this.transactions = transactions;
    }

    /**
     * @brief Returns the previous hash of the block.
     *
     * @return the previous hash of the block.
     */
    public int getPrevHash() {
        return prevHash;
    }

    /**
     * @brief Sets the previous hash based on the previous block.
     *
     * @param prevHash the previous hash.
     */
    public void setPrevHash(int prevHash) {
        this.prevHash = prevHash;
    }

    /**
     * @brief Empty default constructor
     */
    Block(){

    }

    /**
     * @brief Returns a cloned instance of the instantiated block.
     *
     * Kinda useless, no block should EVER be cloned.
     *
     * @return The cloned block.
     */
    public Block clone(){
        return new Block(prevHash,transactions);
    }

    /**
     * @brief A custom equals method which checks if 2 blocks are equal.
     *
     * @param o the object needed for comparison.
     *
     * @return whether the instance is equal with the o object.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Block block = (Block) o;
        return prevHash == block.prevHash && Objects.deepEquals(transactions, block.transactions);
    }

    /**
     * @brief Gets the hash based on the transactions. Transaction 0 will be always needed for variability.
     *
     * @return the hash
     */
    @Override
    public int hashCode() {
        return Objects.hash(prevHash, Arrays.hashCode(transactions),transactions[0].getSource(),transactions[0].getScore());
    }
}
