package hu.afghangoat;

import hu.afghangoat.widgets.StyleWidget;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.input.SAXBuilder;

import javax.swing.text.Style;
import java.awt.*;
import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

/**
 * @class ConfigParser
 * @brief A utility class for getting the right paths for all sorts of configs like images and language settings.
 */
public class ConfigParser {

    /**
     * @brief A getter for the config path.
     */
    public static String getConfigPath() {
        return CONFIG_PATH;
    }

    /**
     * @brief A getter for the image path.
     */
    public static String getImagePath() {
        return IMAGE_PATH;
    }

    /**
     * @brief A getter for the languages path.
     */
    public static String getLangsPath() {
        return LANGS_PATH;
    }

    /**
     * @brief A getter for the game title path.
     *
     * Derived from the config.
     */
    public static String getTitle(){
        return game_title;
    }

    /**
     * @brief The title of the game.
     */
    private static String game_title;

    /**
     * @brief A static string representation of the configuration path.
     */
    private static String CONFIG_PATH = "";

    /**
     * @brief A static string representation of the image path.
     */
    private static String IMAGE_PATH= "";

    /**
     * @brief A static string representation of the languages path.
     */
    private static String LANGS_PATH= "";

    /**
     * @brief The hardcoded name of the top list file.
     */
    public static final String TOP_LIST_FILE_SAVE="top_list.PRIVATE";

    /**
     * @brief Stores the languages in a list.
     */
    private List<LangEntry> languages = new ArrayList<>();

    /**
     * @brief A default language which is derived from the configuration.
     */
    private String DEFAULT_LANG = "en";

    /**
     * @brief Indicates the current language.
     */
    private String curLang=DEFAULT_LANG;

    /**
     * @brief Whether the game uses GPS or flatmap coordinates.
     */
    public static boolean USES_GPS=false;

    /**
     * @brief The scale of the language icon images.
     */
    private double langIconScale=0.3;

    /**
     * @brief A getter for the language icon scale.
     *
     * @return The language icon scale in double representation.
     */
    public double getLangIconScale() {
        return langIconScale;
    }

    /**
     * @brief A getter for the array of the languages.
     *
     * @return All the languages in an array.
     */
    public List<LangEntry> getLanguages(){
        return languages;
    }

    /**
     * @brief A default constructor which finds and sets the config, image and language paths.
     */
    public ConfigParser() {
        Path jarLocation;
        try {
            jarLocation = Paths.get(Main.class.getProtectionDomain()
                    .getCodeSource()
                    .getLocation()
                    .toURI());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        Path configPath = jarLocation.getParent().resolve("config");

        CONFIG_PATH=configPath.toAbsolutePath().toString()+"\\";

        IMAGE_PATH=CONFIG_PATH+"images\\";

        LANGS_PATH=CONFIG_PATH+"langs\\";

        File configFile = new File(CONFIG_PATH + "\\config.xml");

        if (configFile.exists()) {
            try {
                SAXBuilder saxBuilder = new SAXBuilder();
                Document document = saxBuilder.build(configFile);

                Element rootElement = document.getRootElement();

                // Reading languages
                Element languagesElement = rootElement.getChild("languages");
                List<Element> languageElements = languagesElement.getChildren("language");

                for (Element lang : languageElements) {
                    String code = lang.getAttributeValue("code");
                    String name = lang.getText();
                    registerLang(code, name);
                }

                // Reading active language
                String activeLang = rootElement.getChildText("active_lang");
                game_title = rootElement.getChildText("guesser_name");
                setCurrentLang(activeLang);

            } catch (Exception e) {
                System.err.println("Error reading the file: " + e.getMessage());
                e.printStackTrace();
            }
        } else {
            System.out.println("File does not exist: config.xml");
        }
    }

    /**
     * @brief A setter for the current language.
     *
     * @param what The ID of the new language.
     */
    public void setCurrentLang(String what){
        curLang=what;
    }

    /**
     * @brief A getter for the current language.
     *
     * @return The ID of the current language.
     */
    public String getCurrentLang(){
        return curLang;
    }

    /**
     * @brief Adds a new language based on the name and an ID.
     *
     * @param langName The name of the language.
     * @param langTooltip The ID of the language.
     */
    public void registerLang(String langName, String langTooltip){
        if(languages!=null){
            languages.add(new LangEntry(langName,langTooltip));
        }

    }

    /**
     * @brief Returns an array of the themes loaded from a specific file path.
     *
     * @param path The path of the themes XML file.
     *
     * @return An array of the themes the file contained.
     */
    public ArrayList<StyleWidget> getThemes(String path){
        ArrayList<StyleWidget> themes = new ArrayList<>();
        File file = new File(path);
        if (file.exists()) {
            try {
                SAXBuilder saxBuilder = new SAXBuilder();
                Document document = saxBuilder.build(file);

                Element root = document.getRootElement(); // <config>
                List<Element> entries = root.getChildren("entry");

                //themes.add(new StyleWidget(new Color(Integer.parseInt(entry.getChild("background_gradient_1").getChild("r").getText()), Integer.parseInt(entry.getChild("background_gradient_1").getChild("g").getText()), Integer.parseInt(entry.getChild("background_gradient_1").getChild("b").getText())), new Color(Integer.parseInt(entry.getChild("background_gradient_2").getChild("r").getText()), Integer.parseInt(entry.getChild("background_gradient_2").getChild("g").getText()), Integer.parseInt(entry.getChild("background_gradient_2").getChild("b").getText())),new Color(Integer.parseInt(entry.getChild("font_color").getChild("r").getText()), Integer.parseInt(entry.getChild("font_color").getChild("g").getText()), Integer.parseInt(entry.getChild("font_color").getChild("b").getText())),entry.getAttributeValue("key")));

                for (Element entry : entries) {
                    Element bg1 = entry.getChild("background_gradient_1");
                    Element bg2 = entry.getChild("background_gradient_2");
                    Element font = entry.getChild("font_color");

                    if (bg1 == null || bg2 == null || font == null) {
                        //System.out.println("Warning: missing color entry for " + entry.getAttributeValue("key"));
                        System.out.println("Warning: invalid theme in theme file: "+entry.getAttributeValue("key"));
                        continue;
                    }

                    Element bg1r = bg1.getChild("r");
                    Element bg1g = bg1.getChild("g");
                    Element bg1b = bg1.getChild("b");
                    Element bg2r = bg2.getChild("r");
                    Element bg2g = bg2.getChild("g");
                    Element bg2b = bg2.getChild("b");
                    Element ftr = font.getChild("r");
                    Element ftg = font.getChild("g");
                    Element ftb = font.getChild("b");

                    if (bg1r == null || bg1g == null || bg1b == null ||
                            bg2r == null || bg2g == null || bg2b == null ||
                            ftr == null || ftg == null || ftb == null) {
                        //System.out.println("Warning: incomplete color values for " + entry.getAttributeValue("key"));

                        System.out.println("Warning: invalid theme in theme file: "+entry.getAttributeValue("key"));
                        continue;
                    }

                    themes.add(new StyleWidget(
                            new Color(Integer.parseInt(bg1r.getText()), Integer.parseInt(bg1g.getText()), Integer.parseInt(bg1b.getText())),
                            new Color(Integer.parseInt(bg2r.getText()), Integer.parseInt(bg2g.getText()), Integer.parseInt(bg2b.getText())),
                            new Color(Integer.parseInt(ftr.getText()), Integer.parseInt(ftg.getText()), Integer.parseInt(ftb.getText())),
                            entry.getAttributeValue("key")
                    ));
                }

            } catch (Exception e) {
                System.err.println("Error reading the file: " + e.getMessage());
                e.printStackTrace();
            }
        } else {
            System.out.println("File does not exist: " + path);
        }

        return themes;
    }
}
