package junitlab.bank.impl;

import java.util.HashMap;
import java.util.Map;

import junitlab.bank.AccountNotExistsException;
import junitlab.bank.Bank;
import junitlab.bank.NotEnoughFundsException;

/**
 * A szmla mveleteket megvalst msik bank osztly.
 */
public class GreatSavingsBank implements Bank {
	
	/**
	 * A bankban vezetett szmlt megvalst osztly.
	 */
	private class Account {
		
		/**
		 * A szmla szma.
		 */
		private String accountNumber;
		
		/**
		 * A szmla egyenlege. 
		 */
		private long balance;
		
		/**
		 * Adott szmlaszm szmla ltrehozsa. A szmla nyitegyenlege nulla. 
		 * @param accountNumber A szmlaszm.
		 */
		public Account(String accountNumber) {
			this.accountNumber = accountNumber;
			this.balance = 0;
		}
		
		/**
		 * A szmlaszm lekrdezse.
		 * @return A szmla szma.
		 */
		public String getAccountNumber() {
			return accountNumber;
		}
		
		/**
		 * Az egyenleg lekrdezse.
		 * @return A szmla egyenlege.
		 */
		public long getBalance() {
			return balance;
		}

		/**
		 * A megadott sszeg befizetse a szmlra.
		 * @param amount A befizetend sszeg, tetszleges szm lehet.
		 */
		public void deposit(long amount) {
			balance += amount;
		}

		/**
		 * A megadott sszeg kifizetse a szmlrl.
		 * @param amount A kifizetend sszeg, tetszleges szm lehet.
		 * @throws NotEnoughFundsException Ha a szmla egyenlege alacsonyabb mint a megadott sszeg.
		 */
		public void withdraw(long amount) throws NotEnoughFundsException {
			if(amount > balance) {
				throw new NotEnoughFundsException(accountNumber);
			}
			balance -= amount;
		}
	}

	/**
	 * A bankban vezetett szmlk kollekcija.
	 */
	private Map<String, Account> accounts = new HashMap<String, Account>();
	
	/**
	 * Elkeresi a megadott szm szmlt, ha ltezik.
	 * @param accountNumber A keresett szmla szma.
	 * @return A keresett szmla objektum.
	 * @throws AccountNotExistsException Ha nem ltezik az adott szm szmla.
	 */
	private Account getAccount(String accountNumber) throws AccountNotExistsException {
		if(accounts.containsKey(accountNumber)) {
			return accounts.get(accountNumber);
		} else {
			throw new AccountNotExistsException(accountNumber);
		}
	}
	
	/**
	 * A megadott sszeget 100-zal oszthatv kerekti.
	 * @param amount Tetszleges pozitv sszeg.
	 * @return A megadott sszeg a kerekts szablyainak alkalmazsa utn.
	 */
	private long roundAmount(long amount) {
		if(amount <= 0) {
			throw new IllegalArgumentException("Amount must be positive");
		}
		long remainder = amount % 100;
		if(remainder < 50) {
			return amount - remainder;
		} else {
			return amount + 100 - remainder;
		}
	}

	@Override
	public String openAccount() {
		Account newAccount = new Account(String.format("47328000-%08d", accounts.size() + 1));
		accounts.put(newAccount.getAccountNumber(), newAccount);
		return newAccount.getAccountNumber();
	}

	@Override
	public boolean closeAccount(String accountNumber) throws AccountNotExistsException {
		if(getAccount(accountNumber).getBalance() == 0) {
			accounts.remove(accountNumber);
			return true;
		} else {
			return false;
		}
	}

	@Override
	public long getBalance(String accountNumber) throws AccountNotExistsException {
		return getAccount(accountNumber).getBalance();
	}

	@Override
	public void deposit(String accountNumber, long amount) throws AccountNotExistsException {
		getAccount(accountNumber).deposit(roundAmount(amount));
	}

	@Override
	public void withdraw(String accountNumber, long amount) throws AccountNotExistsException, NotEnoughFundsException {
		getAccount(accountNumber).withdraw(roundAmount(amount));
	}

	@Override
	public void transfer(String sourceAccount, String targetAccount, long amount) throws AccountNotExistsException, NotEnoughFundsException {
		if(amount <= 0) {
			throw new IllegalArgumentException("Amount must be positive");
		}
		Account source = getAccount(sourceAccount);
		Account target = getAccount(targetAccount);
		source.withdraw(amount);
		target.deposit(amount);
	}
}
