// Online C++ compiler to run C++ program online
#include <iostream>
#include <optional>
#include <algorithm>
#include <variant>
#include <functional>

/*
A --f--> B --g--> C
       ⟹  g∘f : A → C
       
int -> std::string morfizmus, amiben a data type-ok az objektumok.

Egy funktor leképezést ad kategóriák között, miközben megőrzi a struktúrát:
Egy funktor olyan típus, amiben van egy map/transform művelet.

*/

template<typename T, typename F>
auto fmap(const std::optional<T>& opt, F f) {
    if (opt) {
        return std::optional( f(*opt) );
    }
    return std::optional<decltype(f(*opt))>();
}

using Pair = std::pair<int, std::string>;   // szorzat típus
using Either = std::variant<int, std::string>; // összeg típus

//Maybe monad with optional:
//Ez igazából AZ std::optional<T>

// A "Maybe" típus: egy egyszerű monád C++-ban.
// A Maybe<T> vagy tartalmaz egy értéket (Just), vagy semmit (Nothing).
template<typename T>
struct Maybe {
    public:
    std::optional<T> value; // belső adattag: vagy tartalmaz T-t, vagy üres (std::nullopt)

    // Alap konstruktor – üres Maybe
    Maybe() = default;

    // Konstruktor, ami egy konkrét értéket csomagol be
    Maybe(T v): value(v) {
        
    }

    // "just" — egy Maybe példányt készít, ami tartalmaz értéket
    static Maybe<T> just(T v) { return Maybe(v); }

    // "nothing" — egy Maybe példányt készít, ami üres
    static Maybe<T> nothing() { return {}; }

    // bind (>>=): a monád lényegi művelete
    // Ha van érték, akkor lefuttatja rá a függvényt f,
    // ami egy új Maybe-t ad vissza.
    // Ha nincs érték, akkor továbbviszi a "Nothing" állapotot.
    template<typename F>
    auto bind(F f) const -> decltype(f(value.value())) {
        if (value){
            // Ha van érték, meghívjuk a függvényt f azzal az értékkel.
            return f(value.value());
        }
        
        // Ha nincs érték, akkor egy üres Maybe-t adunk vissza ugyanazzal a típussal.
        using Ret = decltype(f(value.value()));
        return Ret::nothing();
    }
};

//Maybe optional nélkül
template<typename T>
struct Maybe2 {
    public:
    T value; // belső adattag: vagy tartalmaz T-t, vagy üres (std::nullopt)
    bool has_value = false;

    // Alap konstruktor – üres Maybe
    Maybe2() {
        has_value=false;
    }

    // Konstruktor, ami egy konkrét értéket csomagol be
    Maybe2(T v): value(v) {
        has_value=true;
    }
    
    Maybe2(T v, bool val): value(v) {
        has_value=val;
    }

    // "just" — egy Maybe példányt készít, ami tartalmaz értéket
    static Maybe2<T> just(T v) { return Maybe2(v); }

    // "nothing" — egy Maybe példányt készít, ami üres
    static Maybe2<T> nothing() { return Maybe2(); }

    // bind (>>=): a monád lényegi művelete
    // Ha van érték, akkor lefuttatja rá a függvényt f,
    // ami egy új Maybe-t ad vissza.
    // Ha nincs érték, akkor továbbviszi a "Nothing" állapotot.
    template<typename F>
    auto bind(F f) const -> decltype(f(value)) {
        if (has_value){
            // Ha van érték, meghívjuk a függvényt f azzal az értékkel.
            return f(value);
        }
        
        // Ha nincs érték, akkor egy üres Maybe-t adunk vissza ugyanazzal a típussal.
        using Ret = decltype(f(value));
        return Ret::nothing();
    }
};

int main() {
    
    // A "half" függvény: egy számot felez, ha páros.
    // Ha páratlan, akkor "Nothing"-ot ad vissza.
    auto half = [](int x) -> Maybe2<int> {
        if (x % 2 == 0) return Maybe2<int>::just(x / 2);
        return Maybe2<int>::nothing();
    };
    
    // Kiinduló Maybe, ami 20-at tartalmaz.
    Maybe2<int> start = Maybe2<int>::just(20);
    
    // Háromszor egymás után alkalmazzuk a "half" függvényt.
    // Mivel 20 → 10 → 5 → Nothing (mert 5 páratlan), a végeredmény Nothing lesz.
    auto result = start.bind(half).bind(half).bind(half);

    if (result.has_value)
        std::cout << "Result: " << result.value << "\n";
    else
        std::cout << "Nothing\n";
    
    std::optional<int> x = 5;
    std::optional<int> z;

    auto y = fmap(x, [](int v){ return v * 2; });

    if (y)
        std::cout << *y << "\n";  // 10
    else
        std::cout << "empty\n";
        
    
}