#ifndef ELECTRONIC_MODULE_SAVING_HPP
#define ELECTRONIC_MODULE_SAVING_HPP

#include "../electronics_visual_model.hpp"
#include <fstream>
#include "../external/json.hpp"
using json = nlohmann::json;

#include <set>

#include <filesystem>
namespace fs = std::filesystem;

#include "electrical_serializable.hpp"

#include "../GLOBAL_CONSTANTS.hpp"

NLOHMANN_JSON_SERIALIZE_ENUM(ComponentType, {
    {ComponentType::EGND, "GND"},
    {ComponentType::EResistor, "Resistor"},
    {ComponentType::EVoltageSource, "VoltageSource"},
    {ComponentType::ECapacitor, "Capacitor"},
    {ComponentType::ECurrentSource, "CurrentSource"},
    {ComponentType::EDiode, "Diode"},
    {ComponentType::ELogicalPin, "LogicalPin"},
    {ComponentType::ENPNTransistor, "NPNTransistor"},
    {ComponentType::ENullPort, "NullPort"},
    {ComponentType::EOpAmp, "OpAmp"},
	
    {ComponentType::DOrGate, "OrGate"},
    {ComponentType::DAndGate, "AndGate"},
    {ComponentType::DOutputBit, "OutputBit"},
    {ComponentType::DInputBit, "InputBit"},
    {ComponentType::DXorGate, "XorGate"},
    {ComponentType::DNandGate, "NandGate"},
    {ComponentType::DInvertGate, "InvertGate"},
})

class ComponentFactory {
	private:
	
	public:
		static ElectricalComponent* create(ElectricalNodeCollection& electronics, const CustomElectricalComponent& d) {
			ElectricalComponent* created = nullptr;
			switch (d.type) {
				case ComponentType::EGND:
					created=&electronics.addComponent<GND>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;

				case ComponentType::EResistor:
					created=&electronics.addComponent<Resistor>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("resistance")
					);
					break;
					
				case ComponentType::EVoltageSource:
					created=&electronics.addComponent<VoltageSource>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("voltage")
					);
					break;
					
				case ComponentType::ECapacitor:
					created=&electronics.addComponent<Capacitor>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("cap")
					);
					break;
					
				case ComponentType::ECurrentSource:
					created=&electronics.addComponent<CurrentSource>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("cur")
					);
					break;	
					
				case ComponentType::EDiode:
					created=&electronics.addComponent<Diode>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				
				case ComponentType::ELogicalPin:
					created=&electronics.addComponent<LogicalPin>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("v"),
						d.floats.at("iLimit")
					);
					break;
					
				case ComponentType::ENPNTransistor:
					created=&electronics.addComponent<NPNTransistor>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
					
				case ComponentType::ENullPort:
					created=&electronics.addComponent<NullPort>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;	
					
				case ComponentType::EOpAmp:
					created=&electronics.addComponent<OpAmp>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("A")
					);
					break;	
					
				//DIGITAL
				case ComponentType::DOrGate:
					created=&electronics.addComponent<OrGate>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				case ComponentType::DAndGate:
					created=&electronics.addComponent<AndGate>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				case ComponentType::DOutputBit:
					created=&electronics.addComponent<OutputBit>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				case ComponentType::DInputBit:
					created=&electronics.addComponent<InputBit>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins,
						d.floats.at("voltage")
					);
					break;
				case ComponentType::DXorGate:
					created=&electronics.addComponent<XorGate>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				case ComponentType::DNandGate:
					created=&electronics.addComponent<NandGate>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
				case ComponentType::DInvertGate:
					created=&electronics.addComponent<InvertGate>(
						d.position.x,
						d.position.y,
						d.texture.c_str(),
						d.pins
					);
					break;
			}
			return created;
		}
};

void to_json(json& j, const ImVec2& v);

void from_json(const json& j, ImVec2& v);

void to_json(json& j, const CustomElectricalComponent& d);

void from_json(const json& j, CustomElectricalComponent& d);

void to_json(json& j, const PortConnection& c);

void from_json(const json& j, PortConnection& c);

class ElectricComponentLoader{ //for loading .ecf files
	private:
		
	public:
		void saveComponents(const std::string& filename, const std::vector<CustomElectricalComponent>& components);


		std::vector<CustomElectricalComponent> loadComponents(const std::string& filename);
};

class ElectricalComponentCollection{ //Will contain some components of the scene. Can also contain the default component configs.
	private:
		ElectricComponentLoader loader;
		
		std::string fname;
	public:
		std::vector<CustomElectricalComponent*> components;
		
		ElectricalComponentCollection(const std::string& filename);
		
		ElectricalComponentCollection();
		
		~ElectricalComponentCollection();
		
		void loadFrom(const std::string& filename,const std::string& minFileName);
		
		void loadFrom(const std::string& filename); // Overload for scene files (loads all components)
		
		void loadDefaultsElectrical(); //Electrical
		
		void loadDefaultsDigital(); //Digital
		
		static void createDefaultComponents(){
			ElectricComponentLoader loader;
	
			std::vector<CustomElectricalComponent> comps;

			// Add a sample component
			CustomElectricalComponent gnd;
			gnd.type = ComponentType::EGND;
			gnd.position = ImVec2{0, 0};
			gnd.texture = "gnd.png";
			gnd.pins = { ImVec2{50,00}};
			gnd.floats["voltage"] = 0.0f;

			comps.clear();
			comps.push_back(gnd);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"gnd.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent resistor_100_ohm;
			resistor_100_ohm.type = ComponentType::EResistor;
			resistor_100_ohm.position = ImVec2{0, 0};
			resistor_100_ohm.texture = "resistor.png";
			resistor_100_ohm.pins = { ImVec2{0,50}, ImVec2{100,50}};
			resistor_100_ohm.floats["resistance"] = 100.0f;

			comps.clear();
			comps.push_back(resistor_100_ohm);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"resistor_100_ohm.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent resistor_10_ohm;
			resistor_10_ohm.type = ComponentType::EResistor;
			resistor_10_ohm.position = ImVec2{0, 0};
			resistor_10_ohm.texture = "resistor.png";
			resistor_10_ohm.pins = { ImVec2{0,50}, ImVec2{100,50} };
			resistor_10_ohm.floats["resistance"] = 10.0f;

			comps.clear();
			comps.push_back(resistor_10_ohm);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"resistor_10_ohm.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent resistor_1000_ohm;
			resistor_1000_ohm.type = ComponentType::EResistor;
			resistor_1000_ohm.position = ImVec2{0, 0};
			resistor_1000_ohm.texture = "resistor.png";
			resistor_1000_ohm.pins = { ImVec2{0,50}, ImVec2{100,50} };
			resistor_1000_ohm.floats["resistance"] = 1000.0f;

			comps.clear();
			comps.push_back(resistor_1000_ohm);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"resistor_1000_ohm.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent capacitor_1micro_farad;
			capacitor_1micro_farad.type = ComponentType::ECapacitor;
			capacitor_1micro_farad.position = ImVec2{0, 0};
			capacitor_1micro_farad.texture = "capacitor.png";
			capacitor_1micro_farad.pins = { ImVec2{0,50}, ImVec2{100,50} };
			capacitor_1micro_farad.floats["cap"] = 0.000001f;

			comps.clear();
			comps.push_back(capacitor_1micro_farad);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"capacitor_1micro_farad.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent capacitor_1nano_farad;
			capacitor_1nano_farad.type = ComponentType::ECapacitor;
			capacitor_1nano_farad.position = ImVec2{0, 0};
			capacitor_1nano_farad.texture = "capacitor.png";
			capacitor_1nano_farad.pins = { ImVec2{0,50}, ImVec2{100,50} };
			capacitor_1nano_farad.floats["cap"] = 0.000000001f;

			comps.clear();
			comps.push_back(capacitor_1nano_farad);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"capacitor_1nano_farad.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent voltage_source_5v;
			voltage_source_5v.type = ComponentType::EVoltageSource;
			voltage_source_5v.position = ImVec2{0, 0};
			voltage_source_5v.texture = "voltage_source.png";
			voltage_source_5v.pins = { ImVec2{0,50}, ImVec2{100,50} };
			voltage_source_5v.floats["voltage"] = 5.0f;

			comps.clear();
			comps.push_back(voltage_source_5v);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"voltage_source_5v.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent voltage_source_3v;
			voltage_source_3v.type = ComponentType::EVoltageSource;
			voltage_source_3v.position = ImVec2{0, 0};
			voltage_source_3v.texture = "voltage_source.png";
			voltage_source_3v.pins = { ImVec2{0,50}, ImVec2{100,50} };
			voltage_source_3v.floats["voltage"] = 5.0f;

			comps.clear();
			comps.push_back(voltage_source_3v);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"voltage_source_3v.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent voltage_source_3_3v;
			voltage_source_3_3v.type = ComponentType::EVoltageSource;
			voltage_source_3_3v.position = ImVec2{0, 0};
			voltage_source_3_3v.texture = "voltage_source.png";
			voltage_source_3_3v.pins = { ImVec2{0,50}, ImVec2{100,50} };
			voltage_source_3_3v.floats["voltage"] = 5.0f;

			comps.clear();
			comps.push_back(voltage_source_3_3v);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"voltage_source_3_3v.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent current_source_20mA;
			current_source_20mA.type = ComponentType::ECurrentSource;
			current_source_20mA.position = ImVec2{0, 0};
			current_source_20mA.texture = "current_source.png";
			current_source_20mA.pins = { ImVec2{0,50}, ImVec2{100,50} };
			current_source_20mA.floats["cur"] = 0.020f;

			comps.clear();
			comps.push_back(current_source_20mA);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"current_source_20mA.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent ideal_diode;
			ideal_diode.type = ComponentType::EDiode;
			ideal_diode.position = ImVec2{0, 0};
			ideal_diode.texture = "diode.png";
			ideal_diode.pins = { ImVec2{0,50}, ImVec2{100,50} };
			//current_source_20mA.floats["cur"] = 0.020f;

			comps.clear();
			comps.push_back(ideal_diode);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"ideal_diode.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent logical_pin_3_3V;
			logical_pin_3_3V.type = ComponentType::ELogicalPin;
			logical_pin_3_3V.position = ImVec2{0, 0};
			logical_pin_3_3V.texture = "logical_pin.png";
			logical_pin_3_3V.pins = { ImVec2{0,50}, ImVec2{100,50} };
			logical_pin_3_3V.floats["v"] = 3.3f;
			logical_pin_3_3V.floats["iLimit"] = 0.040f;

			comps.clear();
			comps.push_back(logical_pin_3_3V);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"logical_pin_3_3V.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent logical_pin_0V;
			logical_pin_0V.type = ComponentType::ELogicalPin;
			logical_pin_0V.position = ImVec2{0, 0};
			logical_pin_0V.texture = "logical_pin.png";
			logical_pin_0V.pins = { ImVec2{0,50} };
			logical_pin_0V.floats["v"] = 0.0f;
			logical_pin_0V.floats["iLimit"] = 0.040f;

			comps.clear();
			comps.push_back(logical_pin_0V);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"logical_pin_0V.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent npn_transistor;
			npn_transistor.type = ComponentType::ENPNTransistor;
			npn_transistor.position = ImVec2{0, 0};
			npn_transistor.texture = "npn_transistor.png";
			npn_transistor.pins = { ImVec2{0,50}, ImVec2{100,0}, ImVec2{100,100} };
		   // npn_transistor.floats["v"] = 0.0f;
		   // npn_transistor.floats["iLimit"] = 0.040f;

			comps.clear();
			comps.push_back(npn_transistor);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"npn_transistor.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent null_port;
			null_port.type = ComponentType::ENullPort;
			null_port.position = ImVec2{0, 0};
			null_port.texture = "null_port.png";
			null_port.pins = { ImVec2{0,0}};
		   // npn_transistor.floats["v"] = 0.0f;
		   // npn_transistor.floats["iLimit"] = 0.040f;

			comps.clear();
			comps.push_back(null_port);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"null_port.ecf", comps);
			
			// Add a sample component
			CustomElectricalComponent op_amp;
			op_amp.type = ComponentType::EOpAmp;
			op_amp.position = ImVec2{0, 0};
			op_amp.texture = "op_amp.png";
			op_amp.pins = { ImVec2{0,0}, ImVec2{0,100}, ImVec2{100,50}};
			op_amp.floats["A"] = 1e6;

			comps.clear();
			comps.push_back(op_amp);

			// Save to file
			loader.saveComponents(std::string(ELECTRONICS_COMPONENTS_DIRECTORY)+"op_amp.ecf", comps);
			
			//-------		DIGITAL
			//DIGITAL_COMPONENTS_DIRECTORY
			
			
			// Add a sample component
			CustomElectricalComponent or_gate;
			or_gate.type = ComponentType::DOrGate;
			or_gate.position = ImVec2{0, 0};
			or_gate.texture = "or_gate.png";
			or_gate.pins = { ImVec2{100,50}, ImVec2{0,0}, ImVec2{0,100}};
			//npn_transistor.floats["A"] = 1e6;

			comps.clear();
			comps.push_back(or_gate);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"or_gate.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent and_gate;
			and_gate.type = ComponentType::DAndGate;
			and_gate.position = ImVec2{0, 0};
			and_gate.texture = "and_gate.png";
			and_gate.pins = { ImVec2{100,50}, ImVec2{0,0}, ImVec2{0,100}};
			//npn_transistor.floats["A"] = 1e6;

			comps.clear();
			comps.push_back(and_gate);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"and_gate.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent input_bit_high;
			input_bit_high.type = ComponentType::DInputBit;
			input_bit_high.position = ImVec2{0, 0};
			input_bit_high.texture = "input_bit.png";
			input_bit_high.pins = { ImVec2{100,50}};
			input_bit_high.floats["voltage"] = 1.0;

			comps.clear();
			comps.push_back(input_bit_high);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"input_bit_high.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent input_bit_low;
			input_bit_low.type = ComponentType::DInputBit;
			input_bit_low.position = ImVec2{0, 0};
			input_bit_low.texture = "input_bit.png";
			input_bit_low.pins = { ImVec2{100,50}};
			input_bit_low.floats["voltage"] = 0.0;

			comps.clear();
			comps.push_back(input_bit_low);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"input_bit_low.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent output_bit;
			output_bit.type = ComponentType::DOutputBit;
			output_bit.position = ImVec2{0, 0};
			output_bit.texture = "output_bit.png";
			output_bit.pins = { ImVec2{0,50}};
			//input_bit_low.floats["voltage"] = 0.0;

			comps.clear();
			comps.push_back(output_bit);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"output_bit.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent xor_gate;
			xor_gate.type = ComponentType::DXorGate;
			xor_gate.position = ImVec2{0, 0};
			xor_gate.texture = "xor_gate.png";
			xor_gate.pins = { ImVec2{100,50}, ImVec2{0,0}, ImVec2{0,100}};
			//input_bit_low.floats["voltage"] = 0.0;

			comps.clear();
			comps.push_back(xor_gate);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"xor_gate.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent nand_gate;
			nand_gate.type = ComponentType::DNandGate;
			nand_gate.position = ImVec2{0, 0};
			nand_gate.texture = "nand_gate.png";
			nand_gate.pins = { ImVec2{100,50}, ImVec2{0,0}, ImVec2{0,100}};
			//input_bit_low.floats["voltage"] = 0.0;

			comps.clear();
			comps.push_back(nand_gate);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"nand_gate.dcf", comps);
			
			// Add a sample component
			CustomElectricalComponent inverter_gate;
			inverter_gate.type = ComponentType::DInvertGate;
			inverter_gate.position = ImVec2{0, 0};
			inverter_gate.texture = "invert_gate.png";
			inverter_gate.pins = { ImVec2{100,50}, ImVec2{0,50}};
			//input_bit_low.floats["voltage"] = 0.0;

			comps.clear();
			comps.push_back(inverter_gate);

			// Save to file
			loader.saveComponents(std::string(DIGITAL_COMPONENTS_DIRECTORY)+"inverter_gate.dcf", comps);
		}
		
		void save();

		/*void saveTo(const std::string& fn) {
			std::vector<CustomElectricalComponent> copy;
			copy.reserve(components.size());
			for (auto c : components) {
				copy.push_back(*c); // copy each pointed object
			}
			loader.saveComponents(fn, copy);
		}*/
		
		void saveTo(const std::string& fn);

		
		void seedTo(ElectricalNodeCollection& electronics);
};

#endif