#include <iostream>
#include <SFML/Graphics.hpp>

#include "architecture/Bus.hpp"

/*
FUN ideas:

- Change mapper forcibly for games
-

*/

inline static constexpr int FONT_SIZE = 12;
inline static constexpr int LINE_OFFSET = 7;

class NESApp {
	private:
		sf::RenderWindow window;
		sf::Font font;
		//sf::Text text;
		
		std::vector<sf::Text> texts;
		
		sf::Clock clock;
		
		

	public:
		Bus nes;
		
		std::map<uint16_t, std::string> mapAsm;
		
		std::string hex(uint32_t n, uint8_t d)
		{
			std::string s(d, '0');
			for (int i = d - 1; i >= 0; i--, n >>= 4)
				s[i] = "0123456789ABCDEF"[n & 0xF];
			return s;
		};
		
		bool bEmulationRun = false;
		float fResidualTime = 0.0f;
		
		uint8_t nSelectedPalette = 0x00;
	
		NESApp(unsigned int width, unsigned int height): window(sf::VideoMode(width, height), "NES emulator"){
			
			window.setFramerateLimit(30); //limit FPS
			// Load font
			if (!font.loadFromFile("arial.ttf")) {
				std::cerr << "Failed to load font\n";
			}

			// Setup text
			
			// Setup single pixel (1x1 rectangle)
			
			nes.ppu.initGraphics(&window);
		}
		
		~NESApp(){
			
		}
		
		void DrawString(int x, int y,const std::string& what,sf::Color color = sf::Color::White){
			
			sf::Text t;
			t.setFont(font);
			t.setString(what);
			t.setPosition(float(x), float(y));
			
			t.setCharacterSize(FONT_SIZE);
			t.setFillColor(color);
			texts.push_back(t);  // add to list
		}
		
		void DrawRam(int x, int y, uint16_t nAddr, int nRows, int nColumns)
		{
			int nRamX = x, nRamY = y;
			for (int row = 0; row < nRows; row++)
			{
				std::string sOffset = "$" + hex(nAddr, 4) + ":";
				for (int col = 0; col < nColumns; col++)
				{
					sOffset += " " + hex(nes.cpuRead(nAddr, true), 2);
					nAddr += 1;
				}
				DrawString(nRamX, nRamY, sOffset);
				nRamY += FONT_SIZE+LINE_OFFSET;
			}
		}

		void DrawCpu(int x, int y)
		{
			std::string status = "STATUS: ";
			DrawString(x , y , "STATUS:", sf::Color::White);
			DrawString(x  + 64, y, "N", nes.cpu.status & CPU6502::N ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 80, y , "V", nes.cpu.status & CPU6502::V ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 96, y , "-", nes.cpu.status & CPU6502::U ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 112, y , "B", nes.cpu.status & CPU6502::B ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 128, y , "D", nes.cpu.status & CPU6502::D ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 144, y , "I", nes.cpu.status & CPU6502::I ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 160, y , "Z", nes.cpu.status & CPU6502::Z ? sf::Color::Green : sf::Color::Red);
			DrawString(x  + 178, y , "C", nes.cpu.status & CPU6502::C ? sf::Color::Green : sf::Color::Red);
			DrawString(x , y + 10, "PC: $" + hex(nes.cpu.pc, 4));
			DrawString(x , y + 20, "A: $" +  hex(nes.cpu.a, 2) + "  [" + std::to_string(nes.cpu.a) + "]");
			DrawString(x , y + 30, "X: $" +  hex(nes.cpu.x, 2) + "  [" + std::to_string(nes.cpu.x) + "]");
			DrawString(x , y + 40, "Y: $" +  hex(nes.cpu.y, 2) + "  [" + std::to_string(nes.cpu.y) + "]");
			DrawString(x , y + 50, "Stack P: $" + hex(nes.cpu.stkp, 4));
		}

		void DrawCode(int x, int y, int nLines)
		{
			auto it_a = mapAsm.find(nes.cpu.pc);
			int nLineY = (nLines >> 1) * 10 + y;
			if (it_a != mapAsm.end())
			{
				DrawString(x, nLineY, (*it_a).second, sf::Color::Cyan);
				while (nLineY < (nLines * 10) + y)
				{
					nLineY += 10;
					if (++it_a != mapAsm.end())
					{
						DrawString(x, nLineY, (*it_a).second);
					}
				}
			}

			it_a = mapAsm.find(nes.cpu.pc);
			nLineY = (nLines >> 1) * 10 + y;
			if (it_a != mapAsm.end())
			{
				while (nLineY > y)
				{
					nLineY -= 10;
					if (--it_a != mapAsm.end())
					{
						DrawString(x, nLineY, (*it_a).second);
					}
				}
			}
		}

		void run() {
			while (window.isOpen()) {
				handleEvents();
				
				nes.controller[0] = 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::X) ? 0x80 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::Z)? 0x40 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::A) ? 0x20 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::S) ? 0x10 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::Up)? 0x08 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::Down)? 0x04 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::Left)? 0x02 : 0x00;
				nes.controller[0] |= sf::Keyboard::isKeyPressed(sf::Keyboard::Right)? 0x01 : 0x00;
				
				
				if(bEmulationRun ==true){
					float dt = clock.restart().asSeconds();
					if(fResidualTime > 0.0f){
						fResidualTime -=dt;
					} else {
						fResidualTime += (1.0f/60.0f) - dt;
						
						do {nes.clock();} while (!nes.ppu.frame_complete);
						nes.ppu.frame_complete = false;
					}
					
					
				}
				
				draw();
			}
		}
		
		void handleEvents() {
			sf::Event event;
			while (window.pollEvent(event)) {
				if (event.type == sf::Event::Closed) {
					window.close();
				} else if (event.type == sf::Event::KeyPressed) {
					if (event.key.code == sf::Keyboard::C) {
						do {nes.clock();} while (!nes.cpu.complete());
						do {nes.clock();} while (nes.cpu.complete());
					} else if (event.key.code == sf::Keyboard::R) {
						nes.reset();
						
					//Emulate 1 frame
					} else if (event.key.code == sf::Keyboard::F) {
						do {nes.clock();} while (!nes.ppu.frame_complete);
						do {nes.clock();} while (!nes.cpu.complete());
						
						nes.ppu.frame_complete = false;
					} else if (event.key.code == sf::Keyboard::Space) {
						bEmulationRun = !bEmulationRun;
					} else if (event.key.code == sf::Keyboard::P) {
						(++nSelectedPalette) &= 0x07;
					}
					
				}
				
				
			}
		}
		
		//TODO make ppu getScreen and others const

		void draw() {
			window.clear(sf::Color::Black);

			//window.draw(pixel);
			//window.draw(text);
			DrawCpu(516, 2);
			DrawCode(516, 72, 26);
			
			sf::Sprite& screen = nes.ppu.getScreen();
			screen.setScale(2.f, 2.f);
			
			//Draw all palettes
			/*const int nSwatchSize = 6;
			for(int p=0;p<8;p++){ //For each palettes
				for(int i =0;i<4;i++){
					sf::RectangleShape pixel;
					pixel.setSize(sf::Vector2f(nSwatchSize,nSwatchSize));
					pixel.setFillColor(nes.ppu.GetColorFromPalette(p,s));
					pixel.setPosition(516 + p*(nSwatchSize*5) + s*nSwatchSize, 340,);
				}
			}*/
			
			sf::Sprite& PT1 = nes.ppu.getPatternTable(0,nSelectedPalette);
			PT1.setPosition(516,348);
			window.draw(PT1);
			
			sf::Sprite& PT2 = nes.ppu.getPatternTable(1,nSelectedPalette);
			PT2.setPosition(648,348);
			window.draw(PT2);
			
			/*for(uint8_t y = 0; y<30;y++){
				for(uint8_t x=0;x<32;x++){
					uint8_t id =(uint32_t)nes.ppu.tblName[0][y*32+x];
					
					//DrawString(x*16,y*16,std::string(1, (char)id));
					//std::cout << "X: "<< id << std::endl;
					
					//sf::Sprite sprite;
					PT2.setTextureRect(sf::IntRect(
						(id & 0x0F) << 3,
						((id >> 4) & 0x0F) << 3,
						8,
						8
					));
					PT2.setPosition(x*16,y*16);
					PT2.setScale(2.f, 2.f);
					window.draw(PT2);
					
					//DrawPartialSprite(,PT2,( 2);
				}
			}*/
			
			window.draw(screen);
			
			for(auto& t : texts){
				window.draw(t);
			}
			texts.clear();

			window.display();
		}
};

int main(){
	std::shared_ptr<Cartridge> cart = std::make_shared<Cartridge>("nestest.nes");
	
	NESApp app(1000,500);
	
	app.nes.insertCartridge(cart);
	
	app.mapAsm = app.nes.cpu.disassemble(0x0000, 0xFFFF);
	app.nes.reset();
	
	app.run();
	
	return 0;
}