#ifndef PPU2C02_HPP
#define PPU2C02_HPP

#include <cstdint>
#include <memory>
#include <array>

#include <SFML/Graphics.hpp>

#include "Cartridge.hpp"

class PPU2C02{
	private:
		std::shared_ptr<Cartridge> cart;
		
		uint16_t scanline = 0;
		uint16_t cycle = 0;
		
		union{
			struct{
				uint8_t unused: 5;
				uint8_t sprite_overflow: 1;
				uint8_t sprite_zero_hit: 1;
				uint8_t vertical_blank: 1;
			};
			
			uint8_t reg;
		} status;
		
		union{	
			struct{
				uint8_t grayscale : 1;
				uint8_t render_background_left : 1;
				uint8_t render_sprites_left : 1;
				uint8_t render_background : 1;
				uint8_t render_sprites : 1;
				uint8_t enhance_red : 1;
				uint8_t enhance_green : 1;
				uint8_t enhance_blue : 1;
			};

			uint8_t reg;
		} mask;
		
		union PPUCTRL{
			struct{
				uint8_t nametable_x : 1;
				uint8_t nametable_y : 1;
				uint8_t increment_mode : 1;
				uint8_t pattern_sprite : 1;
				uint8_t pattern_background : 1;
				uint8_t sprite_size : 1;
				uint8_t slave_mode : 1; // unused
				uint8_t enable_nmi : 1;
			};

			uint8_t reg;
		} control;
		
		uint8_t address_latch = 0x00;
		uint8_t ppu_data_buffer = 0x00;
		//uint16_t ppu_address = 0x0000;
		
		union loopy_register{
			// Credit to Loopy
			struct{

				uint16_t coarse_x : 5;
				uint16_t coarse_y : 5;
				uint16_t nametable_x : 1;
				uint16_t nametable_y : 1;
				uint16_t fine_y : 3;
				uint16_t unused : 1;
			};

			uint16_t reg = 0x0000;
		};
		
		
		loopy_register vram_addr; // Active "pointer" address into nametable to extract background tile info
		loopy_register tram_addr; // Temporary store of information to be "transferred" into "pointer" at various times
		uint8_t fine_x = 0x00;
		
		uint8_t bg_next_tile_id     = 0x00;
		uint8_t bg_next_tile_attrib = 0x00;
		uint8_t bg_next_tile_lsb    = 0x00;
		uint8_t bg_next_tile_msb    = 0x00;
		uint16_t bg_shifter_pattern_lo = 0x0000;
		uint16_t bg_shifter_pattern_hi = 0x0000;
		uint16_t bg_shifter_attrib_lo  = 0x0000;
		uint16_t bg_shifter_attrib_hi  = 0x0000;
		
		struct sObjectAttributeEntry{
			uint8_t y;
			uint8_t id;
			uint8_t attribute;
			uint8_t x;
		} OAM[64];
			
	public:
		PPU2C02();
		~PPU2C02();
		
		uint8_t tblName[2][1024]; //Name table info, 2Kb (1 name table = 1024 byte)
		
		uint8_t tblPalette[32]; //32 palette entries
		uint8_t tblPattern[2][4096]; //Probably will never be touched.
		
		bool nmi = false; //Non maskable interrup signal
		
		uint8_t* pOAM = (uint8_t*)OAM;
		
		uint8_t oam_addr = 0x00;
		
		void reset();
		
		//Comms with main bus
		uint8_t cpuRead(uint16_t addr, bool rdonly = false);
		void cpuWrite(uint16_t addr, uint8_t data);
		
		//Comms with PPU bus
		uint8_t ppuRead(uint16_t addr, bool rdonly = false);
		void ppuWrite(uint16_t addr, uint8_t data);
		
		void ConnectCartridge(const std::shared_ptr<Cartridge>& cartridge);
		void clock();
		
		bool frame_complete = false;
		
		
		
		std::array<sf::Color, 0x40> palScreen;

		// Window (external owner)
		sf::RenderWindow* window = nullptr;
		sf::Image screenImage;
		sf::Texture screenTexture;
		sf::Sprite screenSprite;

		sf::Image nameTableImage[2];
		sf::Texture nameTableTexture[2];
		sf::Sprite nameTableSprite[2];

		sf::Image patternTableImage[2];
		sf::Texture patternTableTexture[2];
		sf::Sprite patternTableSprite[2];
		
		sf::Sprite& getScreen();
		sf::Sprite& getNameTable(uint8_t i);
		sf::Sprite& getPatternTable(uint8_t i, uint8_t palette);
		
		sf::Color& GetColorFromPaletteRan(uint8_t palette, uint8_t pixel);
		
		void initGraphics(sf::RenderWindow* wnd);

};

#endif