#include "Cartridge.hpp"
#include <iostream>


Cartridge::Cartridge(const std::string& sFileName){
	struct sHeader {
        char name[4];
        uint8_t prg_rom_chunks;
        uint8_t chr_rom_chunks;
        uint8_t mapper1;
        uint8_t mapper2;
        uint8_t prg_ram_size;
        uint8_t tv_system1;
        uint8_t tv_system2;
        char unused[5];
    } __attribute__((packed)) header;
	
	std::ifstream ifs;
	ifs.open(sFileName, std::ifstream::binary);
	if(ifs.is_open()){
		ifs.read(reinterpret_cast<char*>(&header), sizeof(sHeader));

		// Trainer is 512 bytes if bit 2 of mapper1 is set
		if (header.mapper1 & 0x04) {
			ifs.seekg(512, std::ios_base::cur);
		}

		nMapperID = (header.mapper2 & 0xF0) | ((header.mapper1 & 0xF0) >> 4);
		std::cout << "Discovered Mapper with ID: " << int(nMapperID) << std::endl;
		
		
		mirror = (header.mapper1 & 0x01) ? VERTICAL : HORIZONTAL;
		//nMapperID=0;

		nPRGBanks = header.prg_rom_chunks;
		nCHRBanks = header.chr_rom_chunks;
		
		vPRGMem.resize(nPRGBanks * 16384);
		ifs.read(reinterpret_cast<char*>(vPRGMem.data()), vPRGMem.size());

		if (nCHRBanks == 0) {
			// CHR-RAM (8KB)
			vCHRMem.resize(8192);
		} else {
			// CHR-ROM
			vCHRMem.resize(nCHRBanks * 8192);
			ifs.read(reinterpret_cast<char*>(vCHRMem.data()), vCHRMem.size());
		}

		

		// Mapper
		switch (nMapperID) {
			case 0:
				pMapper = std::make_shared<Mapper_000>(nPRGBanks, nCHRBanks);
				break;
			default:
				std::cerr << "Unsupported mapper: " << int(nMapperID) << std::endl;
				break;
		}

		ifs.close();
	} else {
		std::cerr << "Failed to open ROM: " << sFileName << std::endl;
	}
	
	
}

Cartridge::~Cartridge(){
	
}

void Cartridge::reset()
{
	// Note: This does not reset the ROM contents,
	// but does reset the mapper.
	//if (pMapper != nullptr)
	//	pMapper->reset();
}

//Comms with main bus
bool Cartridge::cpuRead(uint16_t addr, uint8_t& data){
	uint32_t mapped_addr = 0;
	
	if(pMapper->cpuMapRead(addr,mapped_addr)){
		data = vPRGMem[mapped_addr];
		return true;
	}
	
	return false;
}

bool Cartridge::cpuWrite(uint16_t addr, uint8_t data){
	uint32_t mapped_addr = 0;
	if(pMapper->cpuMapWrite(addr,mapped_addr)){
		vPRGMem[mapped_addr] = data;
		return true;
	}
	
	return false;
}

//Comms with PPU bus
bool Cartridge::ppuRead(uint16_t addr, uint8_t& data){
	uint32_t mapped_addr = 0;
	
	if(pMapper->ppuMapRead(addr,mapped_addr)){
		data = vCHRMem[mapped_addr];
		return true;
	}
	
	return false;
}

bool Cartridge::ppuWrite(uint16_t addr, uint8_t data){
	uint32_t mapped_addr = 0;
	if(pMapper->ppuMapWrite(addr,mapped_addr)){
		vCHRMem[mapped_addr] = data;
		return true;
	}
	
	return false;
}